<?php
require_once '../config/database.php';

header('Content-Type: application/json');

// Import the enhanced ping functions from monitoring.php
function trySystemPing($host) {
    // Detect OS and use appropriate ping command
    $os = strtoupper(substr(PHP_OS, 0, 3));
    
    if ($os === 'WIN') {
        // Windows ping command (1 packet, 3 second timeout)
        $command = "ping -n 1 -w 3000 " . escapeshellarg($host);
    } else {
        // Unix/Linux ping command (1 packet, 3 second timeout)
        $command = "ping -c 1 -W 3 " . escapeshellarg($host);
    }
    
    $output = array();
    $return_code = 0;
    
    exec($command, $output, $return_code);
    $output_text = implode("\n", $output);
    
    // Check if ping command exists
    if ($return_code === 127 || strpos($output_text, 'command not found') !== false || strpos($output_text, 'not recognized') !== false) {
        return array(
            'success' => false,
            'error' => 'Ping command not available',
            'response_time_ms' => null
        );
    }
    
    // For successful ping (return code 0)
    if ($return_code === 0) {
        $response_time_ms = null;
        
        if ($os === 'WIN') {
            // Windows format parsing
            if (preg_match('/time<1ms/i', $output_text)) {
                $response_time_ms = 0.5;
            } elseif (preg_match('/time[=](\d+(?:\.\d+)?)ms/i', $output_text, $matches)) {
                $response_time_ms = floatval($matches[1]);
            }
        } else {
            // Linux format parsing
            if (preg_match('/time=(\d+(?:\.\d+)?)\s*ms/i', $output_text, $matches)) {
                $response_time_ms = floatval($matches[1]);
            }
        }
        
        if ($response_time_ms === null) {
            $response_time_ms = 1.0;
        }
        
        return array(
            'success' => true,
            'output' => $output_text,
            'response_time_ms' => round($response_time_ms, 1)
        );
    } else {
        return array(
            'success' => false,
            'error' => $output_text,
            'response_time_ms' => null
        );
    }
}

function trySocketPing($host, $timeout = 3) {
    // Test common ports to check connectivity
    $test_ports = [80, 443, 22, 23];
    
    foreach ($test_ports as $port) {
        $start_time = microtime(true);
        
        // Try to connect via socket
        $socket = @fsockopen($host, $port, $errno, $errstr, $timeout);
        
        if ($socket) {
            fclose($socket);
            $response_time = round((microtime(true) - $start_time) * 1000, 1);
            
            return array(
                'success' => true,
                'output' => "Socket connection successful to $host:$port",
                'response_time_ms' => $response_time,
                'method' => 'socket',
                'port' => $port
            );
        }
    }
    
    return array(
        'success' => false,
        'error' => "Cannot connect to $host via socket",
        'response_time_ms' => null
    );
}

function pingHost($host, $timeout = 3) {
    // Remove any protocol prefix
    $host = preg_replace('/^https?:\/\//', '', $host);
    
    // First try system ping command
    $ping_result = trySystemPing($host);
    
    // If system ping fails, fallback to socket ping
    if (!$ping_result['success']) {
        $socket_result = trySocketPing($host, $timeout);
        
        if ($socket_result['success']) {
            return [
                'host' => $host,
                'status' => 'up',
                'response_time' => $socket_result['response_time_ms'],
                'packet_loss' => 0,
                'timestamp' => date('Y-m-d H:i:s'),
                'method' => $socket_result['method'] ?? 'socket'
            ];
        } else {
            return [
                'host' => $host,
                'status' => 'down',
                'response_time' => null,
                'packet_loss' => 100,
                'timestamp' => date('Y-m-d H:i:s'),
                'error' => $socket_result['error']
            ];
        }
    }
    
    // System ping was successful
    return [
        'host' => $host,
        'status' => 'up',
        'response_time' => $ping_result['response_time_ms'],
        'packet_loss' => 0,
        'timestamp' => date('Y-m-d H:i:s'),
        'method' => 'system_ping'
    ];
}

try {
    $database = new Database();
    $db = $database->getConnection();
    
    $action = $_GET['action'] ?? $_POST['action'] ?? '';
    
    switch ($action) {
        case 'ping_single':
            $host = $_GET['host'] ?? '';
            
            if (empty($host)) {
                echo json_encode([
                    'success' => false,
                    'message' => 'Host parameter is required'
                ]);
                break;
            }
            
            $result = pingHost($host);
            
            echo json_encode([
                'success' => true,
                'ping_result' => $result
            ]);
            break;
            
        case 'ping_access_points':
            // Get all Access Points with IP addresses
            $query = "SELECT id, name, ip_address 
                      FROM ftth_items 
                      WHERE item_type_id = 9 
                      AND ip_address IS NOT NULL 
                      AND ip_address != ''
                      ORDER BY name";
            
            $stmt = $db->prepare($query);
            $stmt->execute();
            $access_points = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $ping_results = [];
            
            foreach ($access_points as $ap) {
                $ping_result = pingHost($ap['ip_address']);
                $ping_result['device_id'] = $ap['id'];
                $ping_result['device_name'] = $ap['name'];
                $ping_results[] = $ping_result;
            }
            
            echo json_encode([
                'success' => true,
                'ping_results' => $ping_results,
                'total_devices' => count($access_points),
                'online_devices' => count(array_filter($ping_results, function($r) { return $r['status'] === 'up'; }))
            ]);
            break;
            
        case 'ping_device':
            $device_id = intval($_GET['device_id'] ?? 0);
            
            if (!$device_id) {
                echo json_encode([
                    'success' => false,
                    'message' => 'Device ID is required'
                ]);
                break;
            }
            
            // Get device info
            $query = "SELECT id, name, ip_address, item_type_id 
                      FROM ftth_items 
                      WHERE id = :device_id";
            
            $stmt = $db->prepare($query);
            $stmt->execute([':device_id' => $device_id]);
            $device = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$device) {
                echo json_encode([
                    'success' => false,
                    'message' => 'Device not found'
                ]);
                break;
            }
            
            if (empty($device['ip_address'])) {
                echo json_encode([
                    'success' => false,
                    'message' => 'Device has no IP address configured'
                ]);
                break;
            }
            
            $ping_result = pingHost($device['ip_address']);
            $ping_result['device_id'] = $device['id'];
            $ping_result['device_name'] = $device['name'];
            $ping_result['device_type'] = $device['item_type_id'];
            
            echo json_encode([
                'success' => true,
                'ping_result' => $ping_result
            ]);
            break;
            
        case 'bulk_ping':
            $devices = json_decode($_POST['devices'] ?? '[]', true);
            
            if (empty($devices)) {
                echo json_encode([
                    'success' => false,
                    'message' => 'No devices provided'
                ]);
                break;
            }
            
            $ping_results = [];
            
            foreach ($devices as $device) {
                if (!empty($device['ip_address'])) {
                    $ping_result = pingHost($device['ip_address']);
                    $ping_result['device_id'] = $device['id'];
                    $ping_result['device_name'] = $device['name'];
                    $ping_results[] = $ping_result;
                }
            }
            
            echo json_encode([
                'success' => true,
                'ping_results' => $ping_results,
                'total_tested' => count($ping_results)
            ]);
            break;
            
        default:
            echo json_encode([
                'success' => false,
                'message' => 'Invalid action'
            ]);
            break;
    }
    
} catch (Exception $e) {
    echo json_encode([
        'success' => false,
        'message' => 'Error: ' . $e->getMessage()
    ]);
}
?>
